import {
  BadRequestException,
  Controller,
  Get,
  NotFoundException,
  Param,
  Query,
} from '@nestjs/common';
import { NavigationService } from './navigation.service';
import {
  Graph,
  GraphId,
  GraphNode,
  GraphNodeId,
  PathType,
  NearestReturnType,
} from './navigation.entity';
import {
  ApiTags,
  ApiOperation,
  ApiResponse,
  ApiProduces,
  ApiParam,
  ApiQuery,
} from '@nestjs/swagger';
import { NavigationQueryDto, NearestNodeQueryDto } from './dto/navigation.dto';

@Controller('navigation')
@ApiTags('Pathfinding & Navigation')
export class NavigationController {
  constructor(private readonly navigationService: NavigationService) {}

  /**
   * Get a graph by its ID.
   * @param id The ID of the graph.
   * @param returnType Optional - The type of graph to return (deep or shallow).
   * If 'shallow', returns the graph without nodes. If 'deep', returns the full graph.
   * If not provided, defaults to 'deep'.
   * @returns The graph object.
   */
  @Get('graph/:id')
  @ApiOperation({
    summary: 'Get a graph by its ID',
    description: 'Fetches a graph object based on the provided graph ID.',
  })
  @ApiProduces('application/json')
  @ApiResponse({
    status: 200,
    description: 'Graph object retrieved successfully.',
  })
  @ApiResponse({
    status: 400,
    description: 'Bad request. Graph ID is required or invalid.',
  })
  @ApiParam({ name: 'id', type: String, description: 'Graph ID' })
  @ApiQuery({
    name: 'type',
    type: String,
    description: 'Return type of the graph (deep or shallow)',
    required: false,
    enum: ['deep', 'shallow'],
  })
  async getGraph(
    @Param('id') id: GraphId,
    @Query('type') returnType?: 'deep' | 'shallow',
  ): Promise<Graph | Omit<Graph, 'nodes'>> {
    // Validate id parameter
    if (!id) {
      throw new BadRequestException('Graph ID is required');
    }

    // Fetch the graph using the navigation service
    const graph = await this.navigationService.getGraph(id);
    if (!graph) {
      throw new BadRequestException('Graph not found');
    }
    if (returnType === 'shallow') {
      // Return graph without nodes property
      // eslint-disable-next-line @typescript-eslint/no-unused-vars
      const { nodes, ...shallowGraph } = graph;
      return shallowGraph;
    }
    return graph;
  }

  /**
   * Get a node by its ID.
   * @param nodeId The ID of the node.
   * @returns The node object.
   */
  @Get('node/:nodeId')
  @ApiOperation({
    summary: 'Get a node by its ID',
    description: 'Fetches a node object based on the provided node ID.',
  })
  @ApiProduces('application/json')
  @ApiResponse({
    status: 200,
    description: 'Node object retrieved successfully.',
  })
  @ApiResponse({
    status: 400,
    description: 'Bad request. Node ID is required or invalid.',
  })
  @ApiParam({ name: 'nodeId', type: String, description: 'Node ID' })
  async getNode(@Param('nodeId') nodeId: GraphNodeId): Promise<GraphNode> {
    // Validate nodeId parameter
    if (!nodeId) {
      throw new BadRequestException('Node ID is required');
    }

    // Fetch the node using the navigation service
    const node = await this.navigationService.getNode(nodeId);
    if (!node) {
      throw new BadRequestException('Node not found');
    }
    return node;
  }

  /**
   * Get a path between two nodes.
   * @param from The starting node ID.
   * @param to The destination node ID.
   * @param type Optional - The type of path (deep or shallow).
   * @returns An array of nodes representing the path.
   */
  @Get('path')
  @ApiOperation({
    summary: 'Get a path between two nodes',
    description:
      'Finds a path between two nodes using the A* algorithm. Optionally, specify the type of path (deep or shallow).',
  })
  @ApiProduces('application/json')
  @ApiResponse({
    status: 200,
    description: 'Path found successfully.',
  })
  @ApiResponse({
    status: 400,
    description: 'Bad request. "from" and "to" parameters are required.',
  })
  @ApiResponse({
    status: 400,
    description:
      'Bad request. Invalid "type" parameter. Must be "deep" or "shallow".',
  })
  @ApiResponse({
    status: 404,
    description: 'Path not found between the specified nodes.',
  })
  @ApiQuery({
    name: 'from',
    type: String,
    description: 'Starting node ID',
  })
  @ApiQuery({
    name: 'to',
    type: String,
    description: 'Destination node ID',
  })
  @ApiQuery({
    name: 'type',
    type: String,
    description: 'Type of path to return (deep or shallow)',
    required: false,
    enum: [PathType.DEEP, PathType.SHALLOW],
  })
  async getPath(
    @Query() query: NavigationQueryDto,
  ): Promise<GraphNode[] | { nodes: GraphNodeId[] }> {
    const { from, to, type } = query;

    // Validate required parameters
    if (!from || !to) {
      throw new BadRequestException(
        'Both "from" and "to" parameters are required',
      );
    }

    // Validate type parameter if provided
    if (type && type !== PathType.DEEP && type !== PathType.SHALLOW) {
      throw new BadRequestException(
        'Invalid "type" parameter. Must be "deep" or "shallow".',
      );
    }

    return await this.navigationService.getPath(from, to, type);
  }

  /**
   * Get the nearest node to given coordinates.
   * @param x X coordinate
   * @param y Y coordinate
   * @param z Z coordinate
   * @param returnType Return type: shallow (ID) or deep (full node)
   * @param yThreshold Optional Y threshold for filtering nodes
   */
  @ApiOperation({
    summary: 'Get the nearest node to given coordinates',
    description:
      'Finds the nearest node to the specified coordinates (x, y, z). Optionally, specify the return type (shallow or deep).',
  })
  @ApiProduces('application/json')
  @ApiResponse({
    status: 200,
    description: 'Nearest node found successfully.',
  })
  @ApiResponse({
    status: 400,
    description:
      'Bad request. "x", "y", and "z" query parameters are required.',
  })
  @ApiResponse({
    status: 404,
    description: 'No nodes found near the specified coordinates.',
  })
  @ApiQuery({
    name: 'x',
    type: Number,
    description: 'X coordinate of the search point',
  })
  @ApiQuery({
    name: 'y',
    type: Number,
    description: 'Y coordinate of the search point',
  })
  @ApiQuery({
    name: 'z',
    type: Number,
    description: 'Z coordinate of the search point',
  })
  @ApiQuery({
    name: 'yThreshold',
    type: Number,
    description: 'Optional Y threshold for filtering nodes',
    required: false,
  })
  @ApiQuery({
    name: 'returnType',
    type: String,
    description: 'Return type of the nearest node (shallow or deep)',
    required: false,
    enum: [NearestReturnType.SHALLOW, NearestReturnType.DEEP],
  })
  @Get('nearest')
  async getNearestNode(
    @Query() query: NearestNodeQueryDto,
  ): Promise<string | object> {
    const { x, y, z, yThreshold, returnType } = query;
    if (x === undefined || y === undefined || z === undefined) {
      throw new BadRequestException(
        'x, y, and z query parameters are required',
      );
    }
    const type = returnType || NearestReturnType.SHALLOW;
    const result = (await this.navigationService.getNearestNode(
      Number(x),
      Number(y),
      Number(z),
      yThreshold !== undefined ? Number(yThreshold) : undefined,
      type,
    )) as string | object;
    if (!result) {
      throw new NotFoundException(
        'No nodes found near the specified coordinates.',
      );
    }
    return result;
  }
}
